@description('Prefix used for grouping resources (2-10 characters), e.g. project or product short code')
param prefix string

@description('Environment name that determines the deployment context, e.g. prod, staging, dev')
@minLength(2)
@maxLength(10)
@allowed([
  'prod'
  'staging'
  'dev'
])
param environment string

@description('Hostname for the US App Service')
@minLength(3)
param usAppServiceHostName string

@description('Hostname for the EU App Service')
@minLength(3)
param euAppServiceHostName string

@description('Global domain name')
@minLength(3)
param globalDomain string

@description('US domain name')
@minLength(3)
param usDomain string

@description('EU domain name')
@minLength(3)
param euDomain string

@description('CDN domain name')
@minLength(3)
param cdnDomain string

resource frontDoor 'Microsoft.Cdn/profiles@2024-09-01' = {
  name: '${prefix}-${environment}-global-frontdoor'
  location: 'Global'
  sku: {
    name: 'Standard_AzureFrontDoor'
  }
}

// Endpoints
resource mainEndpoint 'Microsoft.Cdn/profiles/afdEndpoints@2024-09-01' = {
  parent: frontDoor
  name: '${prefix}-${environment}-main-endpoint'
  location: 'Global'
  properties: {
    enabledState: 'Enabled'
  }
}

resource cdnEndpoint 'Microsoft.Cdn/profiles/afdEndpoints@2024-09-01' = {
  parent: frontDoor
  name: '${prefix}-${environment}-cdn-endpoint'
  location: 'Global'
  properties: {
    enabledState: 'Enabled'
  }
}

// Origin Groups
resource globalOriginGroup 'Microsoft.Cdn/profiles/originGroups@2024-09-01' = {
  parent: frontDoor
  name: 'global-origin-group'
  properties: {
    loadBalancingSettings: {
      sampleSize: 4
      successfulSamplesRequired: 3
    }
    healthProbeSettings: {
      probePath: '/'
      probeRequestType: 'HEAD'
      probeProtocol: 'Https'
      probeIntervalInSeconds: 100
    }
    sessionAffinityState: 'Disabled'
  }
}

resource usOriginGroup 'Microsoft.Cdn/profiles/originGroups@2024-09-01' = {
  parent: frontDoor
  name: 'us-origin-group'
  properties: {
    loadBalancingSettings: {
      sampleSize: 4
      successfulSamplesRequired: 3
    }
    healthProbeSettings: {
      probePath: '/'
      probeRequestType: 'HEAD'
      probeProtocol: 'Https'
      probeIntervalInSeconds: 100
    }
    sessionAffinityState: 'Disabled'
  }
}

resource euOriginGroup 'Microsoft.Cdn/profiles/originGroups@2024-09-01' = {
  parent: frontDoor
  name: 'eu-origin-group'
  properties: {
    loadBalancingSettings: {
      sampleSize: 4
      successfulSamplesRequired: 3
    }
    healthProbeSettings: {
      probePath: '/'
      probeRequestType: 'HEAD'
      probeProtocol: 'Https'
      probeIntervalInSeconds: 100
    }
    sessionAffinityState: 'Disabled'
  }
}

// Origins
resource globalGroupUSOrigin 'Microsoft.Cdn/profiles/originGroups/origins@2024-09-01' = {
  parent: globalOriginGroup
  name: 'global-group-us-appservice'
  properties: {
    hostName: usAppServiceHostName
    httpPort: 80
    httpsPort: 443
    originHostHeader: usAppServiceHostName
    priority: 1
    weight: 1000
    enabledState: 'Enabled'
    enforceCertificateNameCheck: true
  }
}

resource globalGroupEUOrigin 'Microsoft.Cdn/profiles/originGroups/origins@2024-09-01' = {
  parent: globalOriginGroup
  name: 'global-group-eu-appservice'
  properties: {
    hostName: euAppServiceHostName
    httpPort: 80
    httpsPort: 443
    originHostHeader: euAppServiceHostName
    priority: 1
    weight: 1000
    enabledState: 'Enabled'
    enforceCertificateNameCheck: true
  }
}

resource usGroupOrigin 'Microsoft.Cdn/profiles/originGroups/origins@2024-09-01' = {
  parent: usOriginGroup
  name: 'us-group-appservice'
  properties: {
    hostName: usAppServiceHostName
    httpPort: 80
    httpsPort: 443
    originHostHeader: usAppServiceHostName
    priority: 1
    weight: 1000
    enabledState: 'Enabled'
    enforceCertificateNameCheck: true
  }
}

resource euGroupOrigin 'Microsoft.Cdn/profiles/originGroups/origins@2024-09-01' = {
  parent: euOriginGroup
  name: 'eu-group-appservice'
  properties: {
    hostName: euAppServiceHostName
    httpPort: 80
    httpsPort: 443
    originHostHeader: euAppServiceHostName
    priority: 1
    weight: 1000
    enabledState: 'Enabled'
    enforceCertificateNameCheck: true
  }
}

// Route configurations
resource mainRouteConfig 'Microsoft.Cdn/profiles/afdEndpoints/routes@2024-09-01' = {
  parent: mainEndpoint
  name: 'main-route'
  properties: {
    originGroup: {
      id: globalOriginGroup.id
    }
    supportedProtocols: [
      'Http'
      'Https'
    ]
    patternsToMatch: [
      '/*'
    ]
    forwardingProtocol: 'HttpsOnly'
    linkToDefaultDomain: 'Enabled'
    httpsRedirect: 'Enabled'
    enabledState: 'Enabled'
    customDomains: [
      { id: globalCustomDomain.id }
      { id: euCustomDomain.id }
      { id: usCustomDomain.id }
    ]
    ruleSets: [
      {
        id: routingRuleSet.id
      }
    ]
  }
  dependsOn: [globalGroupEUOrigin, globalGroupUSOrigin]
}

resource cdnRouteConfig 'Microsoft.Cdn/profiles/afdEndpoints/routes@2024-09-01' = {
  parent: cdnEndpoint
  name: 'cdn-route'
  properties: {
    originGroup: {
      id: globalOriginGroup.id
    }
    supportedProtocols: [
      'Http'
      'Https'
    ]
    patternsToMatch: [
      '/*'
    ]
    forwardingProtocol: 'HttpsOnly'
    linkToDefaultDomain: 'Enabled'
    httpsRedirect: 'Enabled'
    enabledState: 'Enabled'
    cacheConfiguration: {
      queryStringCachingBehavior: 'UseQueryString'
      compressionSettings: {
        isCompressionEnabled: true
        contentTypesToCompress: [
          'text/html'
          'text/css'
          'text/plain'
          'text/xml'
          'text/javascript'
          'application/javascript'
          'application/x-javascript'
          'application/json'
          'application/xml'
          'application/xhtml+xml'
          'application/rss+xml'
          'application/atom+xml'
          'application/x-font-ttf'
          'application/x-font-opentype'
          'application/vnd.ms-fontobject'
          'image/svg+xml'
          'image/x-icon'
          'font/ttf'
          'font/otf'
          'font/eot'
          'font/woff'
          'font/woff2'
          'application/manifest+json'
          'application/vnd.api+json'
          'text/x-component'
          'application/x-font-truetype'
          'application/x-font-woff'
          'application/x-font-woff2'
          'application/x-font-otf'
          'application/x-font-eot'
          'application/x-font-ttc'
        ]
      }
    }
    customDomains: [
      { id: cdnCustomDomain.id }
    ]
    ruleSets: [
      {
        id: corsRuleSet.id
      }
    ]
  }
  dependsOn: [globalGroupEUOrigin, globalGroupUSOrigin]
}

// Rule Sets
resource routingRuleSet 'Microsoft.Cdn/profiles/ruleSets@2024-09-01' = {
  parent: frontDoor
  name: 'regionRoutingRules'
  dependsOn: [globalGroupEUOrigin, globalGroupUSOrigin]
}

resource corsRuleSet 'Microsoft.Cdn/profiles/ruleSets@2024-09-01' = {
  parent: frontDoor
  name: 'allowCorsRules'
  dependsOn: [globalGroupEUOrigin, globalGroupUSOrigin]
}

resource usRoutingRule 'Microsoft.Cdn/profiles/ruleSets/rules@2024-09-01' = {
  parent: routingRuleSet
  name: 'usRouting'
  properties: {
    order: 1
    conditions: [
      {
        name: 'HostName'
        parameters: {
          typeName: 'DeliveryRuleHostNameConditionParameters'
          operator: 'Equal'
          negateCondition: false
          matchValues: [
            usDomain
          ]
          transforms: [
            'Lowercase'
          ]
        }
      }
    ]
    actions: [
      {
        name: 'RouteConfigurationOverride'
        parameters: {
          typeName: 'DeliveryRuleRouteConfigurationOverrideActionParameters'
          originGroupOverride: {
            forwardingProtocol: 'HttpsOnly'
            originGroup: {
              id: usOriginGroup.id
            }
          }
        }
      }
    ]
  }
  dependsOn: [globalGroupUSOrigin]
}

resource euRoutingRule 'Microsoft.Cdn/profiles/ruleSets/rules@2024-09-01' = {
  parent: routingRuleSet
  name: 'euRouting'
  properties: {
    order: 2
    conditions: [
      {
        name: 'HostName'
        parameters: {
          typeName: 'DeliveryRuleHostNameConditionParameters'
          operator: 'Equal'
          negateCondition: false
          matchValues: [
            euDomain
          ]
          transforms: [
            'Lowercase'
          ]
        }
      }
    ]
    actions: [
      {
        name: 'RouteConfigurationOverride'
        parameters: {
          typeName: 'DeliveryRuleRouteConfigurationOverrideActionParameters'
          originGroupOverride: {
            forwardingProtocol: 'HttpsOnly'
            originGroup: {
              id: euOriginGroup.id
            }
          }
        }
      }
    ]
  }
  dependsOn: [globalGroupEUOrigin]
}

resource allowCorsRuleGlobal 'Microsoft.Cdn/profiles/rulesets/rules@2024-09-01' = {
  parent: routingRuleSet
  name: 'allowCorsGlobal'
  properties: {
    order: 100
    conditions: [
      {
        name: 'RequestHeader'
        parameters: {
          typeName: 'DeliveryRuleRequestHeaderConditionParameters'
          operator: 'Equal'
          selector: 'Origin'
          negateCondition: false
          matchValues: [
            'https://${globalDomain}'
          ]
          transforms: [
            'Lowercase'
          ]
        }
      }
    ]
    actions: [
      {
        name: 'ModifyResponseHeader'
        parameters: {
          typeName: 'DeliveryRuleHeaderActionParameters'
          headerAction: 'Overwrite'
          headerName: 'Access-Control-Allow-Origin'
          value: 'https://${globalDomain}'
        }
      }
    ]
    matchProcessingBehavior: 'Continue'
  }
  dependsOn: [
    globalGroupEUOrigin
    globalGroupUSOrigin
  ]
}

resource allowCorsRuleEU 'Microsoft.Cdn/profiles/rulesets/rules@2024-09-01' = {
  parent: corsRuleSet
  name: 'allowCorsEu'
  properties: {
    order: 200
    conditions: [
      {
        name: 'RequestHeader'
        parameters: {
          typeName: 'DeliveryRuleRequestHeaderConditionParameters'
          operator: 'Equal'
          selector: 'Origin'
          negateCondition: false
          matchValues: [
            'https://${euDomain}'
          ]
          transforms: [
            'Lowercase'
          ]
        }
      }
    ]
    actions: [
      {
        name: 'ModifyResponseHeader'
        parameters: {
          typeName: 'DeliveryRuleHeaderActionParameters'
          headerAction: 'Overwrite'
          headerName: 'Access-Control-Allow-Origin'
          value: 'https://${euDomain}'
        }
      }
    ]
    matchProcessingBehavior: 'Continue'
  }
  dependsOn: [
    globalGroupEUOrigin
  ]
}

resource allowCorsRuleUS 'Microsoft.Cdn/profiles/rulesets/rules@2024-09-01' = {
  parent: corsRuleSet
  name: 'allowCorsUs'
  properties: {
    order: 300
    conditions: [
      {
        name: 'RequestHeader'
        parameters: {
          typeName: 'DeliveryRuleRequestHeaderConditionParameters'
          operator: 'Equal'
          selector: 'Origin'
          negateCondition: false
          matchValues: [
            'https://${usDomain}'
          ]
          transforms: [
            'Lowercase'
          ]
        }
      }
    ]
    actions: [
      {
        name: 'ModifyResponseHeader'
        parameters: {
          typeName: 'DeliveryRuleHeaderActionParameters'
          headerAction: 'Overwrite'
          headerName: 'Access-Control-Allow-Origin'
          value: 'https://${usDomain}'
        }
      }
    ]
    matchProcessingBehavior: 'Continue'
  }
  dependsOn: [
    globalGroupUSOrigin
  ]
}

// Custom Domains
resource globalCustomDomain 'Microsoft.Cdn/profiles/customDomains@2024-09-01' = {
  parent: frontDoor
  name: 'global-domain'
  properties: {
    hostName: globalDomain
    tlsSettings: {
      certificateType: 'ManagedCertificate'
      minimumTlsVersion: 'TLS12'
    }
  }
}

resource usCustomDomain 'Microsoft.Cdn/profiles/customDomains@2024-09-01' = {
  parent: frontDoor
  name: 'us-domain'
  properties: {
    hostName: usDomain
    tlsSettings: {
      certificateType: 'ManagedCertificate'
      minimumTlsVersion: 'TLS12'
    }
  }
}

resource euCustomDomain 'Microsoft.Cdn/profiles/customDomains@2024-09-01' = {
  parent: frontDoor
  name: 'eu-domain'
  properties: {
    hostName: euDomain
    tlsSettings: {
      certificateType: 'ManagedCertificate'
      minimumTlsVersion: 'TLS12'
    }
  }
}

resource cdnCustomDomain 'Microsoft.Cdn/profiles/customDomains@2024-09-01' = {
  parent: frontDoor
  name: 'cdn-domain'
  properties: {
    hostName: cdnDomain
    tlsSettings: {
      certificateType: 'ManagedCertificate'
      minimumTlsVersion: 'TLS12'
    }
  }
}
