extension microsoftGraphV1

// Some of the same params as before
param dbAdminSid string
param prefix string
param environment string
param region string
param usDomain string
param euDomain string

// Get existing (global) Graph Service Principal
resource graphSpn 'Microsoft.Graph/servicePrincipals@v1.0' existing = {
  appId: '00000003-0000-0000-c000-000000000000'
}

// Get identity created in Part 1
resource temporaryDeploymentUserAssignedIdentity 'Microsoft.ManagedIdentity/userAssignedIdentities@2023-01-31' existing = {
  name: '${prefix}-${environment}-${region}-temporary-deployment-identity'
}

// App Service Plan
resource appServicePlan 'Microsoft.Web/serverfarms@2024-04-01' = {
  name: '${prefix}-${environment}-${region}-appserviceplan-linux1'
  location: resourceGroup().location
  sku: {
    name: 'B1'
    tier: 'Basic'
  }
  kind: 'linux'
  properties: {
    reserved: true
  }
}

// App Service
resource appService 'Microsoft.Web/sites@2024-04-01' = {
  name: '${prefix}-${environment}-${region}-appservice-web1'
  location: resourceGroup().location
  properties: {
    serverFarmId: appServicePlan.id
    siteConfig: {
      linuxFxVersion: 'DOTNETCORE|9.0'
      alwaysOn: true
    }
  }
  identity: {
    type: 'SystemAssigned'
  }
}

// This is the administrator group for the sql server which includes 
// the managed identity and the specific admin user
resource sqlServerAdminGroup 'Microsoft.Graph/groups@v1.0' = {
  uniqueName: '${prefix}-${environment}-${region}-sqlserver-1-admins'
  displayName: '${prefix}-${environment}-${region}-sqlserver-1 Administrators'
  mailEnabled: false
  mailNickname: '${prefix}-${environment}-${region}-sqlserver-1-admins'
  securityEnabled: true
  owners: [
    dbAdminSid
  ]
  members: [
    dbAdminSid
    temporaryDeploymentUserAssignedIdentity.properties.principalId
  ]
}

// SQL Server
resource sqlServer 'Microsoft.Sql/servers@2024-05-01-preview' = {
  name: '${prefix}-${environment}-${region}-sqlserver-1'
  location: resourceGroup().location
  properties: {
    publicNetworkAccess: 'Enabled'
    administrators: {
      administratorType: 'ActiveDirectory'
      principalType: 'Group'
      login: '${prefix}-${environment}-${region}-sqlserver-1-admin-group'
      sid: sqlServerAdminGroup.id
      tenantId: subscription().tenantId
      azureADOnlyAuthentication: true
    }
    restrictOutboundNetworkAccess: 'Disabled'
    primaryUserAssignedIdentityId: sqlServerIdentity.id
  }
  identity: {
    type: 'UserAssigned'
    userAssignedIdentities: {
      '${sqlServerIdentity.id}': {}
    }
  }
}

// SQL Database
resource database 'Microsoft.Sql/servers/databases@2024-05-01-preview' = {
  parent: sqlServer
  name: '${prefix}-${environment}-${region}-database-web1'
  location: resourceGroup().location
  sku: {
    name: 'Basic'
    tier: 'Basic'
    capacity: 5
  }
  properties: {
    collation: 'SQL_Latin1_General_CP1_CI_AS'
  }
}

// SQL Firewall Rule
resource databaseFirewall 'Microsoft.Sql/servers/firewallRules@2024-05-01-preview' = {
  parent: sqlServer
  name: 'AllowAllWindowsAzureIps'
  properties: {
    startIpAddress: '0.0.0.0'
    endIpAddress: '0.0.0.0'
  }
}

resource sqlServerIdentity 'Microsoft.ManagedIdentity/userAssignedIdentities@2023-01-31' existing = {
  name: '${prefix}-${environment}-${region}-sqlserver-1-identity'
}

// Assign the Directory.Read.All role to the SQL Server identity - this needs to be close to the last step as the identity it takes a while to propagate
resource sqlServerIdentityRoleAssignment 'Microsoft.Graph/appRoleAssignedTo@v1.0' = {
  appRoleId: '7ab1d382-f21e-4acd-a863-ba3e13f7da61'
  principalId: sqlServerIdentity.properties.principalId
  resourceId: graphSpn.id
}

// Add AppService managed identity as db_writer and db_reader 
resource addAppServiceUserToDatabase 'Microsoft.Resources/deploymentScripts@2023-08-01' = {
  name: 'Grant_AppService_Identity_Database_Rights'
  location: resourceGroup().location
  kind: 'AzurePowerShell'
  properties: {
    azPowerShellVersion: '10.0'
    scriptContent: '''
      $sqlServerFqdn = "$env:SQL_SERVER_NAME"
      $sqlDatabaseName = "$env:SQL_DATABASE_NAME"
      $sqlUser = "$env:MANAGED_IDENTITY_NAME"

      # Install SqlServer module
      Install-Module -Name SqlServer -Force -AllowClobber -Scope CurrentUser
      Import-Module SqlServer

      $sqlCmd = @"
      -- Create the user with details retrieved from Entra ID
      CREATE USER [$sqlUser] FROM EXTERNAL PROVIDER

      -- Assign roles to the new user
      ALTER ROLE db_datareader ADD MEMBER [$sqlUser]
      ALTER ROLE db_datawriter ADD MEMBER [$sqlUser]
"@
# Note: the string terminator must not have whitespace before it, therefore it is not indented.

      Write-Host $sqlCmd

      $connectionString = "Server=tcp:${sqlServerFqdn},1433;Initial Catalog=${sqlDatabaseName};Authentication=Active Directory Default;"

      Invoke-Sqlcmd -ConnectionString $connectionString -Query $sqlCmd

    '''
    timeout: 'PT5M'
    retentionInterval: 'PT1H'
    cleanupPreference: 'OnSuccess'
    environmentVariables: [
      {
        name: 'SQL_SERVER_NAME'
        value: sqlServer.properties.fullyQualifiedDomainName
      }
      {
        name: 'SQL_DATABASE_NAME'
        value: database.name
      }
      {
        name: 'MANAGED_IDENTITY_NAME'
        value: appService.name
      }
    ]
  }
  identity: {
    type: 'UserAssigned'
    userAssignedIdentities: {
      '${temporaryDeploymentUserAssignedIdentity.id}': {}
    }
  }
}

// App Service Connection String Configuration
resource appServiceConnectionStrings 'Microsoft.Web/sites/config@2024-04-01' = {
  parent: appService
  name: 'connectionstrings'
  properties: {
    DefaultConnection: {
      value: 'Server=tcp:${sqlServer.properties.fullyQualifiedDomainName},1433;Initial Catalog=${database.name};Authentication=Active Directory Default;'
      type: 'SQLAzure'
    }
  }
}

// Add regions to appSettings
resource appServiceRegionSettings 'Microsoft.Web/sites/config@2022-03-01' = {
  parent: appService
  name: 'appsettings'
  properties: {
    AppRegion: region
    AvailableRegions__eu: euDomain
    AvailableRegions__us: usDomain
  }
}
